const GemType = {
    DIAMOND: 'diamond',
    RUBY: 'ruby',
    EMERALD: 'emerald',
    SAPPHIRE: 'sapphire',
    AMETHYST: 'amethyst',
    TOPAZ: 'topaz',
    CITRINE: 'citrine'
};

const SpecialType = {
    FEATHER: 'feather',
    STAR: 'star',
    KITTY: 'kitty'
};

const GemColors = {
    [GemType.DIAMOND]: { main: '#e2e8f0', light: '#f8fafc', dark: '#94a3b8' }, // White/Silver
    [GemType.RUBY]: { main: '#ef4444', light: '#fca5a5', dark: '#b91c1c' },    // Red
    [GemType.EMERALD]: { main: '#22c55e', light: '#86efac', dark: '#15803d' },  // Green
    [GemType.SAPPHIRE]: { main: '#3b82f6', light: '#93c5fd', dark: '#1d4ed8' },  // Blue
    [GemType.AMETHYST]: { main: '#a855f7', light: '#d8b4fe', dark: '#7e22ce' },  // Purple
    [GemType.TOPAZ]: { main: '#eab308', light: '#fde047', dark: '#a16207' },    // Yellow
    [GemType.CITRINE]: { main: '#f97316', light: '#fdba74', dark: '#c2410c' }   // Orange
};

class GemFactory {
    static createGemSVG(type, colorType = null) {
        // If colorType is provided, use it for colors. Otherwise use type.
        const colorSource = colorType || type;
        const colors = GemColors[colorSource];
        if (!colors) return '';

        // Define gradients for main faces (darker top-left -> lighter bottom-right)
        const defs = `
            <defs>
                <linearGradient id="grad-${colorSource}" x1="0%" y1="0%" x2="100%" y2="100%">
                    <stop offset="0%" style="stop-color:${colors.dark};stop-opacity:0.5" />
                    <stop offset="100%" style="stop-color:${colors.light};stop-opacity:0.5" />
                </linearGradient>
            </defs>
        `;

        let svgContent = '';
        switch (type) {
            // Standard Gems
            case GemType.DIAMOND:
                svgContent = GemFactory._diamondSVG(colors, colorSource);
                break;
            case GemType.RUBY:
                svgContent = GemFactory._rubySVG(colors, colorSource);
                break;
            case GemType.EMERALD:
                svgContent = GemFactory._emeraldSVG(colors, colorSource);
                break;
            case GemType.SAPPHIRE:
                svgContent = GemFactory._sapphireSVG(colors);
                break;
            case GemType.AMETHYST:
                svgContent = GemFactory._amethystSVG(colors);
                break;
            case GemType.TOPAZ:
                svgContent = GemFactory._topazSVG(colors);
                break;
            case GemType.CITRINE:
                svgContent = GemFactory._citrineSVG(colors);
                break;

            // Special Gems
            case SpecialType.FEATHER:
                svgContent = GemFactory._featherSVG(colors);
                break;
            case SpecialType.STAR:
                svgContent = GemFactory._starSVG(colors);
                break;
            case SpecialType.KITTY:
                svgContent = GemFactory._kittySVG(colors);
                break;

            default:
                return '';
        }

        // Inject defs into the SVG string
        return svgContent.replace('<svg viewBox="0 0 100 100">', `<svg viewBox="0 0 100 100">${defs}`);
    }

    static _diamondSVG(c, type) {
        // Round Brilliant-ish (Octagon base)
        return `<svg viewBox="0 0 100 100">
            <!-- Base -->
            <path d="M30 5 L70 5 L95 30 L95 70 L70 95 L30 95 L5 70 L5 30 Z" fill="${c.dark}" />
            <!-- Mid Facets -->
            <path d="M30 5 L5 30 L50 50 Z" fill="${c.light}" /> <!-- Top Left - Brightest -->
            <path d="M70 5 L95 30 L50 50 Z" fill="${c.main}" />
            <path d="M95 70 L70 95 L50 50 Z" fill="${c.dark}" />
            <path d="M5 70 L30 95 L50 50 Z" fill="${c.main}" />
            <!-- Top Table -->
            <path d="M30 5 L70 5 L95 30 L95 70 L70 95 L30 95 L5 70 L5 30 Z" fill="${c.main}" transform="scale(0.6) translate(33,33)" />
            <!-- Gradient Overlay on Table -->
            <path d="M30 5 L70 5 L95 30 L95 70 L70 95 L30 95 L5 70 L5 30 Z" fill="url(#grad-${type})" transform="scale(0.6) translate(33,33)" />
            
            <!-- Highlights -->
            <path d="M30 5 L70 5 L50 50 Z" fill="${c.light}" opacity="0.8" />
            <path d="M5 30 L5 70 L50 50 Z" fill="${c.light}" opacity="0.6" />
            <path d="M95 30 L95 70 L50 50 Z" fill="${c.dark}" opacity="0.4" />
            <path d="M30 95 L70 95 L50 50 Z" fill="${c.dark}" opacity="0.6" />
            <!-- Center Table -->
            <rect x="35" y="35" width="30" height="30" fill="${c.light}" opacity="0.9" />
        </svg>`;
    }

    static _rubySVG(c, type) {
        // Cushion/Square Cut
        return `<svg viewBox="0 0 100 100">
            <!-- Base -->
            <rect x="10" y="10" width="80" height="80" rx="15" fill="${c.dark}" />
            <!-- Facets -->
            <path d="M10 25 L25 25 L25 75 L10 75 Z" fill="${c.light}" /> <!-- Left -->
            <path d="M75 25 L90 25 L90 75 L75 75 Z" fill="${c.dark}" /> <!-- Right -->
            <path d="M25 10 L75 10 L75 25 L25 25 Z" fill="${c.light}" opacity="0.8" /> <!-- Top -->
            <path d="M25 75 L75 75 L75 90 L25 90 Z" fill="${c.dark}" opacity="0.8" /> <!-- Bottom -->
            
            <!-- Center Table -->
            <rect x="25" y="25" width="50" height="50" fill="${c.main}" />
            <!-- Gradient Overlay on Table -->
            <rect x="25" y="25" width="50" height="50" fill="url(#grad-${type})" />

            <!-- Inner Facet Lines -->
            <path d="M25 25 L75 75" stroke="${c.light}" stroke-width="1" opacity="0.4"/>
            <path d="M75 25 L25 75" stroke="${c.light}" stroke-width="1" opacity="0.4"/>
            <!-- Strong Highlight -->
            <path d="M15 15 L35 15 L15 35 Z" fill="white" opacity="0.4" />
        </svg>`;
    }

    static _emeraldSVG(c, type) {
        // Classic Emerald Cut - Rotated 45 degrees
        // Since the whole group is rotated 45deg, "Top" in SVG space becomes "Top-Right" visually.
        // "Left" in SVG space becomes "Top-Left" visually.
        // We want the VISUAL Top-Left to be highlighted.

        return `<svg viewBox="0 0 100 100">
            <g transform="rotate(45 50 50)">
                <!-- Outer Bevel -->
                <path d="M20 5 L80 5 L95 20 L95 80 L80 95 L20 95 L5 80 L5 20 Z" fill="${c.dark}" />
                
                <!-- Inner Step 1 (Facets) -->
                <!-- Increased size of main facets by shrinking the inner area from 15/85 to 25/75 -->
                <path d="M35 25 L65 25 L75 35 L75 65 L65 75 L35 75 L25 65 L25 35 Z" fill="${c.main}" />
                <!-- Gradient Overlay on Main Face -->
                <path d="M35 25 L65 25 L75 35 L75 65 L65 75 L35 75 L25 65 L25 35 Z" fill="url(#grad-${type})" />
                
                <!-- VISUAL TOP-LEFT HIGHLIGHTS -->
                <!-- The SVG 'Left' side (M5 20...) becomes the Visual Top-Left slope -->
                <!-- Extended inner points to match new smaller center -->
                <path d="M5 20 L25 35 L25 65 L5 80 L5 20 Z" fill="${c.light}" opacity="0.9" /> <!-- Visual Top-Left -->
                
                <!-- The SVG 'Top-Left Corner' (M5 20...) becomes Visual Top -->
                <path d="M5 20 L20 5 L35 25 L25 35 L5 20 Z" fill="${c.light}" opacity="0.8" /> <!-- Visual Top -->
                
                <!-- The SVG 'Top' side (M20 5...) becomes Visual Top-Right -->
                <path d="M20 5 L80 5 L65 25 L35 25 L20 5 Z" fill="${c.light}" opacity="0.4" /> <!-- Visual Top-Right -->

                <!-- VISUAL BOTTOM-RIGHT SHADOWS -->
                <!-- The SVG 'Right' side (M95 20...) becomes Visual Bottom-Right slope -->
                <path d="M95 20 L75 35 L75 65 L95 80 L95 20 Z" fill="${c.dark}" opacity="0.6" /> <!-- Visual Bottom-Right -->
                
                <!-- The SVG 'Bottom-Right Corner' becomes Visual Bottom -->
                <path d="M95 80 L80 95 L65 75 L75 65 L95 80 Z" fill="${c.dark}" opacity="0.8" /> <!-- Visual Bottom -->
                
                <!-- The SVG 'Bottom' side becomes Visual Bottom-Left -->
                <path d="M80 95 L20 95 L35 75 L65 75 L80 95 Z" fill="${c.dark}" opacity="0.4" /> <!-- Visual Bottom-Left -->

                <!-- Inner Step 2 (Table) - REMOVED as requested -->
                <!-- <rect x="30" y="30" width="40" height="40" fill="${c.light}" opacity="0.8" /> -->
                
                <!-- Facet Lines -->
                <path d="M5 20 L35 25" stroke="${c.dark}" stroke-width="0.5" opacity="0.3" />
                <path d="M95 20 L65 25" stroke="${c.dark}" stroke-width="0.5" opacity="0.3" />
                <path d="M5 80 L35 75" stroke="${c.dark}" stroke-width="0.5" opacity="0.3" />
                <path d="M95 80 L65 75" stroke="${c.dark}" stroke-width="0.5" opacity="0.3" />
            </g>
        </svg>`;
    }

    static _sapphireSVG(c) {
        // Trillion / Triangle Cut - Scaled up 15%
        // Center is 50,50. Scale 1.15 around center.
        return `<svg viewBox="0 0 100 100">
            <g transform="scale(1.15) translate(-6.5, -6.5)">
                <!-- Base -->
                <path d="M10 20 L90 20 L50 90 Z" fill="${c.dark}" />
                
                <!-- Facets -->
                <!-- Top Facet -->
                <path d="M10 20 L90 20 L70 35 L30 35 Z" fill="${c.light}" />
                
                <!-- Left Facet -->
                <path d="M10 20 L50 90 L50 65 L30 35 Z" fill="${c.main}" />
                
                <!-- Right Facet -->
                <path d="M90 20 L50 90 L50 65 L70 35 Z" fill="${c.dark}" opacity="0.8" />
                
                <!-- Center Table (Flat Face) -->
                <path d="M30 35 L70 35 L50 65 Z" fill="${c.main}" />
                <!-- Gradient Overlay on Table -->
                <path d="M30 35 L70 35 L50 65 Z" fill="url(#grad-${GemType.SAPPHIRE})" />
                
                <!-- Sparkle -->
                <circle cx="40" cy="45" r="3" fill="white" opacity="0.6" />
            </g>
        </svg>`;
    }

    static _amethystSVG(c) {
        // Hexagon / Honeycomb - Extra Wide
        // Width 100 (0 to 100), Height 90 (5 to 95)
        // Mid line at Y=50, X goes from 0 to 100
        return `<svg viewBox="0 0 100 100">
            <!-- Base -->
            <path d="M25 5 L75 5 L100 50 L75 95 L25 95 L0 50 Z" fill="${c.dark}" />
            
            <!-- Facets -->
            
            <!-- Top Facet -->
            <path d="M25 5 L75 5 L60 35 L40 35 Z" fill="${c.light}" opacity="0.9" />
            
            <!-- Top Right Facet -->
            <path d="M75 5 L100 50 L80 50 L60 35 Z" fill="${c.main}" />
            
            <!-- Bottom Right Facet -->
            <path d="M100 50 L75 95 L60 65 L80 50 Z" fill="${c.dark}" opacity="0.8" />
            
            <!-- Bottom Facet -->
            <path d="M75 95 L25 95 L40 65 L60 65 Z" fill="${c.dark}" opacity="0.6" />
            
            <!-- Bottom Left Facet -->
            <path d="M25 95 L0 50 L20 50 L40 65 Z" fill="${c.dark}" opacity="0.4" />
            
            <!-- Top Left Facet -->
            <path d="M0 50 L25 5 L40 35 L20 50 Z" fill="${c.light}" opacity="0.6" />
            
            <!-- Center Star / Table -->
            <path d="M40 35 L60 35 L80 50 L60 65 L40 65 L20 50 Z" fill="${c.main}" />
            
            <!-- Star Highlight -->
            <path d="M40 35 L50 50 L20 50 Z" fill="${c.light}" opacity="0.5" />
            <path d="M60 35 L50 50 L40 35 Z" fill="${c.light}" opacity="0.3" />
        </svg>`;
    }

    static _topazSVG(c) {
        // Round Brilliant
        return `<svg viewBox="0 0 100 100">
            <circle cx="50" cy="50" r="45" fill="${c.dark}" />
            <circle cx="50" cy="50" r="35" fill="${c.main}" />
            <!-- Star Pattern -->
            <path d="M50 15 L60 50 L50 85 L40 50 Z" fill="${c.light}" opacity="0.5" />
            <path d="M15 50 L50 60 L85 50 L50 40 Z" fill="${c.light}" opacity="0.5" />
            <!-- Table -->
            <circle cx="50" cy="50" r="20" fill="${c.light}" opacity="0.3" />
            <!-- Highlight -->
            <circle cx="35" cy="35" r="8" fill="white" opacity="0.4" />
        </svg>`;
    }

    static _citrineSVG(c) {
        // Pear / Teardrop - Round Bottom
        return `<svg viewBox="0 0 100 100">
            <!-- Base: Pointed Top (50,5), Wide Round Bottom -->
            <!-- Using Arc for bottom: M10 60 A 40 40 0 0 0 90 60 L 50 5 Z -->
            <path d="M15 55 Q 10 85 50 95 Q 90 85 85 55 L 50 5 Z" fill="${c.dark}" />
            
            <!-- Main Body Facet -->
            <path d="M50 5 L85 55 Q 88 75 50 90 Q 12 75 15 55 Z" fill="${c.main}" />
            
            <!-- Top Facets (Highlights) -->
            <path d="M50 5 L15 55 L50 40 Z" fill="${c.light}" opacity="0.8" /> <!-- Top Left -->
            <path d="M50 5 L85 55 L50 40 Z" fill="${c.light}" opacity="0.5" /> <!-- Top Right -->
            
            <!-- Bottom Facet (Depth) -->
            <path d="M15 55 Q 15 80 50 90 Q 85 80 85 55 L 50 70 Z" fill="${c.dark}" opacity="0.4" />

            <!-- Table -->
            <path d="M50 40 L70 55 L50 70 L30 55 Z" fill="${c.light}" opacity="0.4" />
            
            <!-- Strong Highlight on Curve -->
            <ellipse cx="30" cy="50" rx="5" ry="15" transform="rotate(-15 30 50)" fill="white" opacity="0.5" />
        </svg>`;
    }

    static _featherSVG(c) {
        return `<svg viewBox="0 0 100 100">
            <!-- Gemstone Feather -->
            <!-- Base Shape (Darker back) -->
            <path d="M25 85 Q 20 60 35 40 Q 45 25 55 10 Q 45 30 40 50 Q 35 70 25 85 Z" fill="${c.dark}" />
            <path d="M30 90 Q 50 70 60 50 Q 70 30 75 25 Q 80 40 70 60 Q 60 80 30 90 Z" fill="${c.dark}" />

            <!-- Left Facets (Main & Light) -->
            <path d="M25 85 Q 35 70 40 50 Q 45 30 55 10 L 50 50 L 25 85 Z" fill="${c.main}" />
            <path d="M25 85 L 30 65 L 40 50 L 50 50 L 25 85 Z" fill="${c.light}" opacity="0.6" />
            
            <!-- Right Facets (Main & Dark Shadow) -->
            <path d="M30 90 L 50 50 L 75 25 Q 70 30 60 50 Q 50 70 30 90 Z" fill="${c.main}" />
            <path d="M50 50 L 60 50 L 70 60 L 30 90 Z" fill="${c.dark}" opacity="0.4" />

            <!-- Central Spine (Ridge, not white line) -->
            <!-- Created by the meeting of left and right facets, emphasized with a highlight edge -->
            <path d="M25 85 Q 40 60 55 10" fill="none" stroke="${c.light}" stroke-width="1" opacity="0.8" />
            
            <!-- Angular Cuts/Notches (Gem style) -->
            <path d="M30 65 L 35 60 L 32 55 Z" fill="${c.dark}" />
            <path d="M60 50 L 65 55 L 68 50 Z" fill="${c.dark}" />
        </svg>`;
    }

    static _starSVG(c) {
        return `<svg viewBox="0 0 100 100">
            <!-- 5 Point Star -->
            <polygon points="50,5 61,35 95,35 68,55 79,85 50,65 21,85 32,55 5,35 39,35" fill="${c.main}" stroke="${c.dark}" stroke-width="2" />
            <!-- Inner Bevel -->
            <polygon points="50,20 55,40 75,40 60,50 65,70 50,55 35,70 40,50 25,40 45,40" fill="${c.light}" opacity="0.6" />
            <!-- Center -->
            <circle cx="50" cy="50" r="10" fill="${c.light}" />
        </svg>`;
    }

    static _kittySVG(c) {
        return `<svg viewBox="0 0 100 100">
            <!-- Iconic Kitty Head Shape -->
            <!-- Combined path for ears and head for a clean silhouette -->
            <path d="M20 35 L 25 10 L 45 25 L 55 25 L 75 10 L 80 35 Q 90 45 90 60 Q 90 90 50 90 Q 10 90 10 60 Q 10 45 20 35 Z" fill="${c.main}" stroke="${c.dark}" stroke-width="2" />
            
            <!-- Inner Ear Details (Simple triangles) -->
            <path d="M28 35 L 30 20 L 40 30 Z" fill="${c.light}" opacity="0.6" />
            <path d="M72 35 L 70 20 L 60 30 Z" fill="${c.light}" opacity="0.6" />
            
            <!-- Soft Highlight on forehead -->
            <ellipse cx="50" cy="45" rx="20" ry="10" fill="${c.light}" opacity="0.3" />
        </svg>`;
    }
}
